/*
 * Decompiled with CFR 0.152.
 */
package com.github.mizosoft.methanol.internal.decoder;

import com.github.mizosoft.methanol.decoder.AsyncDecoder;
import com.github.mizosoft.methanol.internal.decoder.InflaterUtils;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.zip.CRC32;
import java.util.zip.Inflater;
import java.util.zip.ZipException;

final class GzipDecoder
implements AsyncDecoder {
    static final String ENCODING = "gzip";
    private static final int GZIP_MAGIC = 35615;
    private static final int CM_DEFLATE = 8;
    private static final int HEADER_SIZE = 10;
    private static final int HEADER_SKIPPED_SIZE = 6;
    private static final int TRAILER_SIZE = 8;
    private static final int TEMP_BUFFER_SIZE = 10;
    private static final int MIN_GZIP_STREAM_SIZE = 20;
    private static final int BYTE_MASK = 255;
    private static final int SHORT_MASK = 65535;
    private static final long INT_MASK = 0xFFFFFFFFL;
    private final Inflater inflater = new Inflater(true);
    private final ByteBuffer tempBuffer = ByteBuffer.allocate(10).order(ByteOrder.LITTLE_ENDIAN);
    private final CRC32 crc = new CRC32();
    private boolean computeHcrc;
    private State state = State.BEGIN;
    private int flags;
    private int fieldLength;
    private int fieldPosition;

    GzipDecoder() {
    }

    @Override
    public String encoding() {
        return ENCODING;
    }

    @Override
    public void decode(AsyncDecoder.ByteSource source, AsyncDecoder.ByteSink sink) throws IOException {
        IOException failedToReadConcatenatedHeader = null;
        block13: while (this.state != State.END) {
            switch (this.state) {
                case BEGIN: {
                    this.state = State.HEADER.prepare(this);
                }
                case HEADER: {
                    if (source.remaining() < 10L) break block13;
                    this.readHeader(source);
                    this.state = State.fromNextFlag(this);
                    continue block13;
                }
                case FLG_EXTRA_LEN: {
                    if (source.remaining() < 2L) break block13;
                    this.fieldLength = this.getUShort(source);
                    this.state = State.FLG_EXTRA_DATA.prepare(this);
                }
                case FLG_EXTRA_DATA: {
                    if (!this.trySkipExtraField(source)) break block13;
                    this.state = State.fromNextFlag(this);
                    continue block13;
                }
                case FLG_ZERO_TERMINATED_FIELD: {
                    if (!this.tryConsumeToZeroByte(source)) break block13;
                    this.state = State.fromNextFlag(this);
                    continue block13;
                }
                case FLG_HCRC: {
                    if (source.remaining() < 2L) break block13;
                    long crc16 = this.crc.getValue() & 0xFFFFL;
                    GzipDecoder.checkValue(crc16, this.getUShort(source), "corrupt gzip header");
                    this.state = State.DEFLATED.prepare(this);
                }
                case DEFLATED: {
                    InflaterUtils.inflateSourceWithChecksum(this.inflater, source, sink, this.crc);
                    if (!this.inflater.finished()) break block13;
                    this.state = State.TRAILER.prepare(this);
                }
                case TRAILER: {
                    if (source.remaining() < 8L) break block13;
                    this.readTrailer(source);
                    this.state = State.POSSIBLY_CONCATENATED_HEADER;
                }
                case POSSIBLY_CONCATENATED_HEADER: {
                    if (source.remaining() < 20L) {
                        if (!source.finalSource()) break block13;
                        this.state = State.END;
                        continue block13;
                    }
                    State.HEADER.prepare(this);
                    try {
                        this.readHeader(source);
                        this.state = State.fromNextFlag(this);
                    }
                    catch (IOException e) {
                        failedToReadConcatenatedHeader = e;
                        this.state = State.END;
                    }
                    continue block13;
                }
                default: {
                    throw new AssertionError((Object)("Unexpected state: " + String.valueOf((Object)this.state)));
                }
            }
        }
        if (this.state == State.END && source.hasRemaining()) {
            IOException streamFinishedPrematurely = new IOException("Gzip stream finished prematurely");
            if (failedToReadConcatenatedHeader != null) {
                streamFinishedPrematurely.addSuppressed(failedToReadConcatenatedHeader);
            }
            throw streamFinishedPrematurely;
        }
        if (this.state != State.END && source.finalSource()) {
            throw new EOFException("Unexpected end of gzip stream");
        }
    }

    @Override
    public void close() {
        this.inflater.end();
    }

    private ByteBuffer read(AsyncDecoder.ByteSource source, int byteCount) {
        assert (source.remaining() >= (long)byteCount);
        source.pullBytes(this.tempBuffer.rewind().limit(byteCount));
        if (this.computeHcrc) {
            this.crc.update(this.tempBuffer.rewind());
        }
        return this.tempBuffer.rewind();
    }

    private int getUByte(AsyncDecoder.ByteSource source) {
        return this.read(source, 1).get() & 0xFF;
    }

    private int getUShort(AsyncDecoder.ByteSource source) {
        return this.read(source, 2).getShort() & 0xFFFF;
    }

    private long getUInt(AsyncDecoder.ByteSource source) {
        return (long)this.read(source, 4).getInt() & 0xFFFFFFFFL;
    }

    private void readHeader(AsyncDecoder.ByteSource source) throws IOException {
        GzipDecoder.checkValue(35615L, this.getUShort(source), "Not in gzip format");
        GzipDecoder.checkValue(8L, this.getUByte(source), "Unsupported compression method");
        int flags = this.getUByte(source);
        if (Flag.RESERVED.isEnabled(flags)) {
            throw new ZipException(String.format("Unsupported flags: %#x", flags));
        }
        if (!Flag.HCRC.isEnabled(flags)) {
            this.computeHcrc = false;
        }
        this.flags = flags;
        this.read(source, 6);
    }

    private void readTrailer(AsyncDecoder.ByteSource source) throws IOException {
        GzipDecoder.checkValue(this.crc.getValue(), this.getUInt(source), "Corrupt gzip stream (CRC32)");
        GzipDecoder.checkValue(this.inflater.getBytesWritten() & 0xFFFFFFFFL, this.getUInt(source), "Corrupt gzip stream (ISIZE)");
    }

    private boolean trySkipExtraField(AsyncDecoder.ByteSource source) {
        while (source.hasRemaining() && this.fieldPosition < this.fieldLength) {
            ByteBuffer buffer = source.currentSource();
            int skipped = Math.min(buffer.remaining(), this.fieldLength - this.fieldPosition);
            int updatedPosition = buffer.position() + skipped;
            if (this.computeHcrc) {
                int originalLimit = buffer.limit();
                this.crc.update(buffer.limit(updatedPosition));
                buffer.limit(originalLimit);
            } else {
                buffer.position(updatedPosition);
            }
            this.fieldPosition += skipped;
        }
        return this.fieldPosition >= this.fieldLength;
    }

    private boolean tryConsumeToZeroByte(AsyncDecoder.ByteSource source) {
        while (source.hasRemaining()) {
            ByteBuffer buffer = source.currentSource();
            if (this.computeHcrc) {
                buffer.mark();
            }
            int zeroPosition = -1;
            while (buffer.hasRemaining()) {
                if (buffer.get() != 0) continue;
                zeroPosition = buffer.position() - 1;
                break;
            }
            int originalLimit = -1;
            if (zeroPosition >= 0) {
                originalLimit = buffer.limit();
                buffer.limit(zeroPosition + 1);
            }
            if (this.computeHcrc) {
                this.crc.update(buffer.reset());
            }
            if (zeroPosition < 0) continue;
            buffer.limit(originalLimit);
            return true;
        }
        return false;
    }

    private static void checkValue(long expected, long found, String msg) throws IOException {
        if (expected != found) {
            throw new ZipException(String.format("%s; expected: %#x, found: %#x", msg, expected, found));
        }
    }

    private static enum Flag {
        RESERVED(224, State.END){

            @Override
            boolean isEnabled(int flags) {
                return (flags & this.value) != 0;
            }
        }
        ,
        EXTRA(4, State.FLG_EXTRA_LEN),
        NAME(8, State.FLG_ZERO_TERMINATED_FIELD),
        COMMENT(16, State.FLG_ZERO_TERMINATED_FIELD),
        HCRC(2, State.FLG_HCRC),
        TEXT(1, State.DEFLATED),
        NONE(0, State.DEFLATED);

        final int value;
        final State state;

        private Flag(int value, State state) {
            this.value = value;
            this.state = state;
        }

        boolean isEnabled(int flags) {
            return (flags & this.value) == this.value;
        }

        int clear(int flags) {
            return flags & ~this.value;
        }

        State prepareState(GzipDecoder decoder) {
            return this.state.prepare(decoder);
        }

        static Flag nextEnabled(int flags) {
            for (Flag flag : Flag.values()) {
                if (!flag.isEnabled(flags)) continue;
                return flag;
            }
            throw new AssertionError((Object)("Couldn't get next Flag for: " + Integer.toHexString(flags)));
        }
    }

    private static enum State {
        BEGIN{

            @Override
            void onPrepare(GzipDecoder decoder) {
            }
        }
        ,
        HEADER{

            @Override
            void onPrepare(GzipDecoder decoder) {
                decoder.crc.reset();
                decoder.computeHcrc = true;
            }
        }
        ,
        FLG_EXTRA_LEN{

            @Override
            void onPrepare(GzipDecoder decoder) {
                decoder.fieldLength = 0;
            }
        }
        ,
        FLG_EXTRA_DATA{

            @Override
            void onPrepare(GzipDecoder decoder) {
                decoder.fieldPosition = 0;
            }
        }
        ,
        FLG_ZERO_TERMINATED_FIELD{

            @Override
            void onPrepare(GzipDecoder decoder) {
            }
        }
        ,
        FLG_HCRC{

            @Override
            void onPrepare(GzipDecoder decoder) {
                decoder.computeHcrc = false;
            }
        }
        ,
        DEFLATED{

            @Override
            void onPrepare(GzipDecoder decoder) {
                decoder.inflater.reset();
                decoder.crc.reset();
            }
        }
        ,
        TRAILER{

            @Override
            void onPrepare(GzipDecoder decoder) {
            }
        }
        ,
        POSSIBLY_CONCATENATED_HEADER{

            @Override
            void onPrepare(GzipDecoder decoder) {
                HEADER.onPrepare(decoder);
            }
        }
        ,
        END{

            @Override
            void onPrepare(GzipDecoder decoder) {
            }
        };


        abstract void onPrepare(GzipDecoder var1);

        @CanIgnoreReturnValue
        final State prepare(GzipDecoder decoder) {
            this.onPrepare(decoder);
            return this;
        }

        private static State fromNextFlag(GzipDecoder decoder) {
            Flag flag = Flag.nextEnabled(decoder.flags);
            decoder.flags = flag.clear(decoder.flags);
            return flag.prepareState(decoder);
        }
    }
}

