/*
 * Decompiled with CFR 0.152.
 */
package com.github.mizosoft.methanol.internal.cache;

import com.github.mizosoft.methanol.CacheControl;
import com.github.mizosoft.methanol.MutableRequest;
import com.github.mizosoft.methanol.ResponseBuilder;
import com.github.mizosoft.methanol.internal.cache.CacheResponse;
import com.github.mizosoft.methanol.internal.cache.HttpDates;
import com.github.mizosoft.methanol.internal.util.Compare;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Optional;

public final class CacheStrategy {
    private static final Duration ONE_DAY = Duration.ofDays(1L);
    private final Duration age;
    private final Duration freshness;
    private final Duration staleness;
    private final Optional<LocalDateTime> lastModified;
    private final Optional<String> etag;
    private final boolean usesHeuristicFreshness;
    private final Servability servability;

    /*
     * Enabled aggressive block sorting
     */
    private CacheStrategy(Factory factory, Instant now) {
        CacheControl requestCacheControl = factory.requestCacheControl;
        CacheControl responseCacheControl = factory.responseCacheControl;
        this.age = factory.computeAge(now);
        this.usesHeuristicFreshness = factory.usesHeuristicFreshness();
        this.lastModified = factory.lastModified;
        this.etag = factory.cacheResponseHeaders.firstValue("ETag");
        this.freshness = factory.computeFreshnessLifetime().minus(this.age);
        this.staleness = this.freshness.negated();
        if (requestCacheControl.noCache() || responseCacheControl.noCache()) {
            this.servability = Servability.NONE;
            return;
        }
        if (!this.freshness.isNegative() && !this.freshness.isZero()) {
            this.servability = requestCacheControl.minFresh().isEmpty() || this.freshness.compareTo(requestCacheControl.minFresh().get()) >= 0 ? Servability.FRESH : Servability.NONE;
            return;
        }
        if (responseCacheControl.mustRevalidate()) {
            this.servability = Servability.NONE;
            return;
        }
        if (requestCacheControl.maxStale().isPresent() || requestCacheControl.anyMaxStale()) {
            this.servability = requestCacheControl.maxStale().map(maxStale -> this.staleness.compareTo((Duration)maxStale) <= 0).orElse(false) != false || requestCacheControl.anyMaxStale() ? Servability.STALE : Servability.NONE;
            return;
        }
        if (responseCacheControl.staleWhileRevalidate().isPresent()) {
            this.servability = responseCacheControl.staleWhileRevalidate().map(staleWhileRevalidate -> this.staleness.compareTo((Duration)staleWhileRevalidate) <= 0).orElse(false) != false ? Servability.STALE_WHILE_REVALIDATING : Servability.NONE;
            return;
        }
        if (!requestCacheControl.staleIfError().isPresent() && !responseCacheControl.staleIfError().isPresent()) {
            this.servability = Servability.NONE;
            return;
        }
        this.servability = requestCacheControl.staleIfError().or(responseCacheControl::staleIfError).map(staleIfError -> this.staleness.compareTo((Duration)staleIfError) <= 0).orElse(false) != false ? Servability.STALE_ON_ERROR : Servability.NONE;
    }

    public boolean isCacheResponseServable() {
        switch (this.servability) {
            case FRESH: 
            case STALE: 
            case STALE_WHILE_REVALIDATING: {
                return true;
            }
        }
        return false;
    }

    public boolean isCacheResponseServableOnError() {
        return this.servability == Servability.STALE_ON_ERROR;
    }

    public boolean requiresBackgroundRevalidation() {
        return this.servability == Servability.STALE_WHILE_REVALIDATING;
    }

    public void addCacheHeaders(ResponseBuilder<?> builder) {
        builder.setHeader("Age", Long.toString(this.age.toSeconds()));
        if (this.freshness.isNegative()) {
            builder.header("Warning", "110 - \"Response is Stale\"");
        }
        if (this.usesHeuristicFreshness && this.age.compareTo(ONE_DAY) > 0) {
            builder.header("Warning", "113 - \"Heuristic Expiration\"");
        }
    }

    public HttpRequest conditionalize(HttpRequest request) {
        return this.etag.isEmpty() && this.lastModified.isEmpty() ? request : MutableRequest.copyOf(request).apply(conditionalRequest -> {
            this.etag.ifPresent(etag -> conditionalRequest.setHeader("If-None-Match", (String)etag));
            this.lastModified.ifPresent(lastModified -> conditionalRequest.setHeader("If-Modified-Since", HttpDates.formatHttpDate(lastModified)));
        }).toImmutableRequest();
    }

    static CacheStrategy create(CacheControl requestCacheControl, CacheResponse cacheResponse, Instant now) {
        return new Factory(requestCacheControl, cacheResponse).create(now);
    }

    private static final class Factory {
        final CacheControl requestCacheControl;
        final CacheControl responseCacheControl;
        final Instant timeRequestSent;
        final Instant timeResponseReceived;
        final HttpHeaders cacheResponseHeaders;
        final LocalDateTime date;
        final Optional<Duration> maxAge;
        final Optional<LocalDateTime> expires;
        final Optional<LocalDateTime> lastModified;

        Factory(CacheControl requestCacheControl, CacheResponse cacheResponse) {
            this.requestCacheControl = requestCacheControl;
            this.responseCacheControl = CacheControl.parse(cacheResponse.get().headers());
            this.timeRequestSent = cacheResponse.get().timeRequestSent();
            this.timeResponseReceived = cacheResponse.get().timeResponseReceived();
            this.cacheResponseHeaders = cacheResponse.get().headers();
            this.maxAge = requestCacheControl.maxAge().or(this.responseCacheControl::maxAge);
            this.lastModified = cacheResponse.get().headers().firstValue("Last-Modified").flatMap(HttpDates::tryParseHttpDate);
            this.date = cacheResponse.get().headers().firstValue("Date").flatMap(HttpDates::tryParseHttpDate).orElseGet(() -> HttpDates.toUtcDateTime(this.timeResponseReceived));
            this.expires = cacheResponse.get().headers().firstValue("Expires").map(expiresValues -> HttpDates.tryParseHttpDate(expiresValues).orElse(LocalDateTime.MIN));
        }

        Duration computeAge(Instant now) {
            Duration age = this.cacheResponseHeaders.firstValue("Age").flatMap(HttpDates::tryParseDeltaSeconds).orElse(Duration.ZERO);
            Duration apparentAge = Compare.max(Duration.between(this.date.toInstant(ZoneOffset.UTC), this.timeResponseReceived), Duration.ZERO);
            Duration responseDelay = Duration.between(this.timeRequestSent, this.timeResponseReceived);
            Duration correctedAge = age.plus(responseDelay);
            Duration correctedInitialAge = Compare.max(apparentAge, correctedAge);
            Duration residentTime = Duration.between(this.timeResponseReceived, now);
            return correctedInitialAge.plus(residentTime);
        }

        Duration computeFreshnessLifetime() {
            return this.maxAge.or(() -> this.expires.map(expires -> Compare.max(Duration.between(this.date, expires), Duration.ZERO))).orElseGet(this::computeHeuristicFreshnessLifetime);
        }

        Duration computeHeuristicFreshnessLifetime() {
            return this.lastModified.map(lastModified -> Compare.max(Duration.between(lastModified, this.date), Duration.ZERO)).orElse(Duration.ZERO).dividedBy(10L);
        }

        boolean usesHeuristicFreshness() {
            return this.maxAge.isEmpty() && this.expires.isEmpty();
        }

        CacheStrategy create(Instant now) {
            return new CacheStrategy(this, now);
        }
    }

    private static enum Servability {
        NONE,
        FRESH,
        STALE,
        STALE_WHILE_REVALIDATING,
        STALE_ON_ERROR;

    }
}

